/*
 * @(#)OMBshObject.java  1.0  23. M�rz 2004
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.oo.objectmodelbsh;

import ch.hslu.cm.oo.objectmodel.*;
import bsh.*;
import java.util.*;
/**
 * Simulates the structure and behavior of an object, which can be scripted
 * using BeanShell.
 * 
 * 
 * @author Werner Randelshofer
 * @version 1.0 23. M�rz 2004  Created.
 */
public class OMBshObject extends OMObject {
    /**
     * This value is used to observe simulated object activations.
     * The value is incremented on entry of a method and decremented on exit
     * of a method.
     * A value of 0 indicates that no method is in progress.
     * A value greater than 0 indicates that a method is in progress.
     */
    private int activations = 0;
    
    /** Creates a new instance. */
    public OMBshObject() {
    }
    
    public OMBshClass getSimulatedClassBsh() {
        return (OMBshClass) getSimulatedClass();
    }
    private ObjectModelBsh getSimulationBsh() {
        return (ObjectModelBsh) getSimulation();
    }
    /**
     * Invokes a simulated method on this simulated object.
     */
    public Object invokeMethod(String name, Object[] args)
    throws EvalError {
        return invokeMethod(null, name, args);
    }
    public Object invokeMethod(OMLink link, String name, Object[] args)
    throws EvalError {
        fireMethodEntered(link, name, true);
        Object response;
        try {
            InstanceNameSpace nameSpace = new InstanceNameSpace(this, getSimulatedClassBsh());
            Interpreter bsh = new Interpreter(getSimulationBsh().getConsole(), nameSpace);
            response = nameSpace.invokeMethod(name, args, bsh);
            fireMethodExited(link, response, false);
            return response;
        } catch (EvalError e) {
            fireMethodExited(link, e, true);
            throw e;
        }
    }
    /**
     * This operation must be invoked by all SimulatedObjects on entry in a simulated
     * method.
     * <p>
     * This method sends a methodEntered message to all registered
     * SimulatedObjectListeners.
     *
     * @param link The link that was used to send the message.
     * @param message The message that was sent to invoke the method. Must be a String in
     * UML notation.
     * @param visualizeInternalProcessing Set this parameter to true, if the
     * internal processing of this method should be visualized.
     */
    public void fireMethodEntered(OMLink link, String message, boolean visualizeInternalProcessing) {
        OMBshLink lnk = (OMBshLink) link;
        if (lnk != null) {
            try {
                Thread.sleep(Math.max(1, getSimulation().getDelay() / 8));
            } catch (InterruptedException e) {
                // ignore
            }
            
            lnk.activate();
            message = getName()
            +"."
            +message;
            ((OMBshObject) link.getConnected(this)).fireTransmittingMessage(link, message);
            lnk.sendMessageTo(message, this, false);
        }
        activations++;
        //getSimulation().getCallStack().addElement(this);
        
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, message, visualizeInternalProcessing, false);
                    ((OMObjectListener)listeners[i+1]).methodEntered(event);
                }
            }
        }
        long delay;
        if (visualizeInternalProcessing) {
            delay = getSimulation().getDelay() / 2;
        } else {
            delay = getSimulation().getDelay() / 4;
        }
        try {
            Thread.sleep(Math.max(1, delay));
        } catch (InterruptedException e) {}
    }
    /**
     * This operation or fireMethodExited(OMLink) must be invoked
     * by all SimulatedObjects on exit of a simulated method.
     * 
     * 
     * @param link The link that was used to send the message.
     */
    public void fireMethodExited(OMLink link) {
        fireMethodExited(link, Primitive.VOID, false);
    }
    /**
     * This operation or or fireMethodExited(OMLink, String) must be invoked by all SimulatedObjects on exit of a simulated
     * method.
     * 
     * 
     * @param link The link that was used to send the message.
     * @param response The response of the method. Must be a String in
     * UML notation containing the message and the return value of the method.
     */
    public void fireMethodExited(OMLink link, Object response, boolean isException) {
        OMBshLink lnk = (OMBshLink) link;
        
        fireMethodExited0(link, response, isException);
        fireTransmittingResponse(link, response, isException);
        if (lnk != null) {
            lnk.sendMessageTo(response, link.getAssociate(this), true);
            // XXX this is broken: should provide message not the response
            ((OMBshObject) lnk.getAssociate(this)).fireMessageTransmitted(link, response);
        } else {
            if (getSimulation().getDelay() != 0) {
                
                try { Thread.sleep(getSimulation().getDelay() / 4);} catch (InterruptedException e) {}
                if (response != Primitive.VOID && response != Void.TYPE) {
                    try { Thread.sleep(getSimulation().getDelay());} catch (InterruptedException e) {}
                }
                if ((response instanceof String) && ((String) response).indexOf('\n') != -1) {
                    try { Thread.sleep(getSimulation().getDelay());} catch (InterruptedException e) {}
                }
            }
        }
        activations--;
        fireResponseTransmitted(link, response, isException);
    }
    
    private void fireMethodExited0(OMLink link, Object response, boolean isException) {
        //getSimulation().getCallStack().removeElementAt(getSimulation().getCallStack().size() - 1);
        
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, response, false, isException);
                    OMObjectListener l = ((OMObjectListener)listeners[i+1]);
                    l.methodExited(event);
                }
            }
            
        }
        if (link != null) link.deactivate();
    }
    
    private void fireTransmittingMessage(OMLink link, String message) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, message, true, false);
                    ((OMObjectListener)listeners[i+1]).transmittingMessage(event);
                }
            }
        }
    }
    private void fireMessageTransmitted(OMLink link, Object message) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, message, false, false);
                    ((OMObjectListener)listeners[i+1]).messageTransmitted(event);
                }
            }
        }
    }
    private void fireTransmittingResponse(OMLink link, Object response, boolean isException) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, response, true, isException);
                    ((OMObjectListener)listeners[i+1]).transmittingResponse(event);
                }
            }
        }
    }
    private void fireResponseTransmitted(OMLink link, Object response, boolean isException) {
        if (listenerList != null) {
            OMObjectEvent event = null;
            
            // Guaranteed to return a non-null array
            Object[] listeners = listenerList.getListenerList();
            // Process the listeners last to first, notifying
            // those that are interested in this event
            for (int i = listeners.length-2; i>=0; i-=2) {
                if (listeners[i]==OMObjectListener.class) {
                    // Lazily create the event:
                    if (event == null)
                        event = new OMObjectEvent(this, link, response, false, isException);
                    ((OMObjectListener)listeners[i+1]).responseTransmitted(event);
                }
            }
        }
    }
    
    public OMBshObject clone() {
        OMBshObject that = (OMBshObject) super.clone();
        return that;
    }
    
    /**
     * Returns the maximum allowed number of links with the given name.
     * Returns -1, if the object does not support links of the given name.
     */
    public int getMaxLinkCount(OMBshClass declaringClass, String linkName) {
        if (declaringClass == null || getSimulatedClass() == null) {
            return Integer.MAX_VALUE;
        }
        OMAssociation a = declaringClass.findAssociationForLinkName(linkName);
        if (a != null) {
            int multiplicity = (a.getMemberLabel().equals(linkName)) ? a.getMemberMultiplicity() : a.getOwnerMultiplicity();
            return (multiplicity < 0) ? Integer.MAX_VALUE : multiplicity;
        }
        return -1;
    }
    public OMObject getLinkedObject(OMBshClass declaringClass, String linkName, int index) {
        OMAssociation a = declaringClass.findAssociationForLinkName(linkName);
        ArrayList c = getLinks(a);
        return (c.size() <= index) ? null : (OMBshObject) ((OMLink) c.get(index)).getConnected(this);
    }
    public void setLinkedObject(OMBshClass declaringClass, String linkName, int index, OMObject o) {
        OMAssociation a = declaringClass.findAssociationForLinkName(linkName);
        ArrayList c = getLinks(a);
        if (c.size() > index) {
            getSimulation().remove((OMLink) c.get(index));
        }
            OMBidirectionalLink sbl = new OMBidirectionalLink();
            sbl.setAssociation(a);
            sbl.setStart(this);
            sbl.setEnd(o);
            getSimulation().add(sbl);
    }
}
